<?php
/**
 * Quick database check for rebate system
 */

require_once("../../conn.php");

echo "<h2>Database Check for Rebate System</h2>";

// Check if bet_rebates table exists
$tableCheck = $conn->query("SHOW TABLES LIKE 'bet_rebates'");
if ($tableCheck->num_rows == 0) {
    echo "<div style='background: #f8d7da; padding: 10px; border-radius: 5px; color: #721c24;'>";
    echo "<strong>ERROR:</strong> bet_rebates table does not exist!";
    echo "</div>";
    exit;
}

echo "<div style='background: #d4edda; padding: 10px; border-radius: 5px; color: #155724;'>";
echo "<strong>✓ SUCCESS:</strong> bet_rebates table exists";
echo "</div>";

// Check table structure
echo "<h3>Table Structure</h3>";
$structureQuery = "DESCRIBE bet_rebates";
$structureResult = $conn->query($structureQuery);

$hasIsPaid = false;
$hasPaidAt = false;
$hasVipLevel = false;

if ($structureResult) {
    echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
    echo "<tr style='background: #f2f2f2;'>";
    echo "<th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th>";
    echo "</tr>";
    
    while ($row = $structureResult->fetch_assoc()) {
        $field = $row['Field'];
        if ($field == 'is_paid') $hasIsPaid = true;
        if ($field == 'paid_at') $hasPaidAt = true;
        if ($field == 'vip_level') $hasVipLevel = true;
        
        echo "<tr>";
        echo "<td>" . htmlspecialchars($field) . "</td>";
        echo "<td>" . htmlspecialchars($row['Type']) . "</td>";
        echo "<td>" . htmlspecialchars($row['Null']) . "</td>";
        echo "<td>" . htmlspecialchars($row['Key']) . "</td>";
        echo "<td>" . htmlspecialchars($row['Default']) . "</td>";
        echo "<td>" . htmlspecialchars($row['Extra']) . "</td>";
        echo "</tr>";
    }
    echo "</table>";
}

// Check required columns
echo "<h3>Required Columns Check</h3>";
echo "<ul>";
echo "<li><strong>is_paid:</strong> " . ($hasIsPaid ? "<span style='color: green;'>✓ EXISTS</span>" : "<span style='color: red;'>✗ MISSING</span>") . "</li>";
echo "<li><strong>paid_at:</strong> " . ($hasPaidAt ? "<span style='color: green;'>✓ EXISTS</span>" : "<span style='color: red;'>✗ MISSING</span>") . "</li>";
echo "<li><strong>vip_level:</strong> " . ($hasVipLevel ? "<span style='color: green;'>✓ EXISTS</span>" : "<span style='color: red;'>✗ MISSING</span>") . "</li>";
echo "</ul>";

// Check if VIP table exists
$vipTableCheck = $conn->query("SHOW TABLES LIKE 'vip'");
if ($vipTableCheck->num_rows > 0) {
    echo "<div style='background: #d4edda; padding: 10px; border-radius: 5px; color: #155724;'>";
    echo "<strong>✓ SUCCESS:</strong> vip table exists";
    echo "</div>";
    
    // Check VIP table structure
    echo "<h3>VIP Table Structure</h3>";
    $vipStructureQuery = "DESCRIBE vip";
    $vipStructureResult = $conn->query($vipStructureQuery);
    
    if ($vipStructureResult) {
        echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
        echo "<tr style='background: #f2f2f2;'>";
        echo "<th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th>";
        echo "</tr>";
        
        while ($row = $vipStructureResult->fetch_assoc()) {
            echo "<tr>";
            echo "<td>" . htmlspecialchars($row['Field']) . "</td>";
            echo "<td>" . htmlspecialchars($row['Type']) . "</td>";
            echo "<td>" . htmlspecialchars($row['Null']) . "</td>";
            echo "<td>" . htmlspecialchars($row['Key']) . "</td>";
            echo "<td>" . htmlspecialchars($row['Default']) . "</td>";
            echo "<td>" . htmlspecialchars($row['Extra']) . "</td>";
            echo "</tr>";
        }
        echo "</table>";
    }
} else {
    echo "<div style='background: #fff3cd; padding: 10px; border-radius: 5px; color: #856404;'>";
    echo "<strong>⚠️ WARNING:</strong> vip table does not exist. VIP-based rebates will not work.";
    echo "</div>";
}

// Check sample data
echo "<h3>Sample Data Check</h3>";
$sampleQuery = "SELECT COUNT(*) as total_rebates, 
                       COUNT(DISTINCT user_id) as unique_users,
                       SUM(rebate_amount) as total_amount,
                       MIN(created_at) as oldest_rebate,
                       MAX(created_at) as newest_rebate
                FROM bet_rebates";
$sampleResult = $conn->query($sampleQuery);

if ($sampleResult) {
    $sampleRow = $sampleResult->fetch_assoc();
    echo "<ul>";
    echo "<li><strong>Total Rebates:</strong> " . number_format($sampleRow['total_rebates']) . "</li>";
    echo "<li><strong>Unique Users:</strong> " . number_format($sampleRow['unique_users']) . "</li>";
    echo "<li><strong>Total Amount:</strong> ₹" . number_format($sampleRow['total_amount'], 2) . "</li>";
    echo "<li><strong>Oldest Rebate:</strong> " . htmlspecialchars($sampleRow['oldest_rebate']) . "</li>";
    echo "<li><strong>Newest Rebate:</strong> " . htmlspecialchars($sampleRow['newest_rebate']) . "</li>";
    echo "</ul>";
}

// Check for unpaid rebates
if ($hasIsPaid) {
    echo "<h3>Unpaid Rebates Check</h3>";
    $unpaidQuery = "SELECT DATE(created_at) as rebate_date,
                           COUNT(*) as unpaid_count,
                           SUM(rebate_amount) as unpaid_amount
                    FROM bet_rebates 
                    WHERE is_paid IS NULL OR is_paid = 0
                    GROUP BY DATE(created_at)
                    ORDER BY rebate_date DESC
                    LIMIT 10";
    $unpaidResult = $conn->query($unpaidQuery);
    
    if ($unpaidResult && $unpaidResult->num_rows > 0) {
        echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
        echo "<tr style='background: #f2f2f2;'>";
        echo "<th>Date</th><th>Unpaid Count</th><th>Unpaid Amount</th>";
        echo "</tr>";
        
        while ($row = $unpaidResult->fetch_assoc()) {
            echo "<tr>";
            echo "<td>" . htmlspecialchars($row['rebate_date']) . "</td>";
            echo "<td>" . number_format($row['unpaid_count']) . "</td>";
            echo "<td>₹" . number_format($row['unpaid_amount'], 2) . "</td>";
            echo "</tr>";
        }
        echo "</table>";
    } else {
        echo "<p>No unpaid rebates found.</p>";
    }
}

// Show actions needed
echo "<h3>Actions Needed</h3>";
$actionsNeeded = [];

if (!$hasIsPaid || !$hasPaidAt) {
    $actionsNeeded[] = "Run migrate_rebate_table.php to add missing columns";
}

if (!$hasVipLevel) {
    $actionsNeeded[] = "Run migrate_rebate_table.php to add vip_level column";
}

if ($vipTableCheck->num_rows == 0) {
    $actionsNeeded[] = "Create VIP table or ensure VIP system is properly configured";
}

if (empty($actionsNeeded)) {
    echo "<div style='background: #d4edda; padding: 10px; border-radius: 5px; color: #155724;'>";
    echo "<strong>✓ ALL GOOD:</strong> Database is ready for the rebate system!";
    echo "</div>";
} else {
    echo "<div style='background: #f8d7da; padding: 10px; border-radius: 5px; color: #721c24;'>";
    echo "<strong>Actions Required:</strong>";
    echo "<ul>";
    foreach ($actionsNeeded as $action) {
        echo "<li>$action</li>";
    }
    echo "</ul>";
    echo "</div>";
}

$conn->close();
?>
